
import { API_CONFIG, ACTIONS } from './shared.js';

// background.js
let detachedWindowId = null;
let isPinned = false;

// Helper function to check if sidePanel API is available
function isSidePanelSupported() {
  return chrome?.sidePanel !== undefined;
}

// Helper function to find focused window
async function getFocusedTab() {
  try {
    const tabs = await chrome.tabs.query({ active: true, lastFocusedWindow: true });
    return tabs[0];
  } catch (error) {
    console.error('Error getting focused tab:', error);
    return null;
  }
}



// Single message listener for all actions
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  console.log('Received message in background:', message);

  if (!message || typeof message.action !== 'string') {
    console.error('Invalid message format');
    sendResponse({ success: false, error: 'Invalid message format' });
    return false;
  }

  switch (message.action) {
    case 'getSharedSecret':
      console.log('Handling getSharedSecret request');
      fetchSharedSecret()
        .then(secret => {
          console.log('Secret fetched successfully');
          sendResponse({ success: true, secret });
        })
        .catch(error => {
          console.error('Error fetching secret:', error);
          sendResponse({ success: false, error: error.message });
        });
      return true; // Keep channel open for async response

    case 'getPinState':
      sendResponse({ isPinned });
      break;

    case 'updatePinState':
      isPinned = message.isPinned;
      updatePinIcon().catch(console.error);
      sendResponse({ success: true });
      break;

    case 'createDetachedWindow':
      handleDetachedWindow(message, sendResponse);
      return true;

    case 'togglePin':
      handleTogglePin(sendResponse);
      return true;

    case 'getLastActiveTab':
      handleGetLastActiveTab(sendResponse);
      return true;

    default:
      console.warn(`Unknown action received: ${message.action}`);
      sendResponse({ success: false, error: 'Unknown action' });
      return false;
  }

  return true;
});



// async function handleDetachedWindow(message, sendResponse) {
//   try {
//     const tab = await getFocusedTab();
//     if (tab) {
//       await chrome.storage.local.set({
//         'lastActiveTab': {
//           id: tab.id,
//           url: tab.url,
//           title: tab.title
//         }
//       });
//     }

//     const { width, height, left, top } = message.options;
//     chrome.windows.create({
//       url: chrome.runtime.getURL('popup.html') + '?detached=true',
//       type: 'popup',
//       width: width,
//       height: height,
//       left: left,
//       top: top,
//       focused: true
//     }, (window) => {
//       if (chrome.runtime.lastError) {
//         console.error('Error creating window:', chrome.runtime.lastError);
//         sendResponse({ success: false, error: chrome.runtime.lastError.message });
//       } else {
//         detachedWindowId = window.id;
//         sendResponse({ success: true });
//       }
//     });
//   } catch (error) {
//     console.error('Error handling detached window:', error);
//     sendResponse({ success: false, error: error.message });
//   }
// }


// Utility functions
async function handleDetachedWindow(message, sendResponse) {
  try {
    const tab = await getFocusedTab();
    if (tab) {
      await chrome.storage.local.set({
        'lastActiveTab': {
          id: tab.id,
          url: tab.url,
          title: tab.title
        }
      });
    }

    const { width, height, left, top } = message.options;
    chrome.windows.create({
      url: chrome.runtime.getURL('popup.html') + '?detached=true',
      type: 'popup',
      width: width,
      height: height,
      left: left,
      top: top,
      focused: true
    }, (window) => {
      if (chrome.runtime.lastError) {
        sendResponse({ success: false, error: chrome.runtime.lastError.message });
      } else {
        detachedWindowId = window.id;
        sendResponse({ success: true });
      }
    });
  } catch (error) {
    sendResponse({ success: false, error: error.message });
  }
}




function handleTogglePin(sendResponse) {
  if (isSidePanelSupported()) {
    chrome.windows.getCurrent(async (window) => {
      try {
        if (!window || !window.id) {
          throw new Error('No active window available');
        }

        const result = await toggleSidePanel();
        sendResponse({
          success: true,
          isPinned: result.isPinned
        });
      } catch (error) {
        console.error('Side panel toggle error:', error);
        sendResponse({
          success: false,
          error: error.message
        });
      }
    });
  } else {
    sendResponse({
      success: false,
      error: 'Side panel not supported in this browser'
    });
  }
  return true; // Keep the message channel open
}

function handleGetLastActiveTab(sendResponse) {
  chrome.storage.local.get(['lastActiveTab'], (result) => {
    sendResponse({ tab: result.lastActiveTab });
  });
}

// Track window closure
chrome.windows.onRemoved.addListener((windowId) => {
  if (windowId === detachedWindowId) {
    detachedWindowId = null;
    chrome.storage.local.remove('lastActiveTab');
  }
});

// Reset window state on startup
chrome.runtime.onStartup.addListener(() => {
  detachedWindowId = null;
  chrome.storage.local.remove('lastActiveTab');
});

async function updatePinIcon() {
  // Skip icon update to avoid errors
  // Just track the state
  console.log('Pin state updated:', isPinned);
}


// Side panel functionality

async function toggleSidePanel() {
  try {
    if (!isSidePanelSupported()) {
      throw new Error('Side panel not supported');
    }

    const currentWindow = await chrome.windows.getCurrent();

    if (!currentWindow || !currentWindow.id) {
      throw new Error('No active window found');
    }

    if (isPinned) {
      await chrome.sidePanel.close({
        windowId: currentWindow.id
      });
    } else {
      await chrome.sidePanel.open({
        windowId: currentWindow.id
      });
    }

    isPinned = !isPinned;
    // Skip icon update, just return state
    return { isPinned };
  } catch (error) {
    console.error('Error toggling side panel:', error);
    throw error;
  }
}


// Initialize side panel state if supported
if (isSidePanelSupported()) {
  // Set up keyboard shortcut
  chrome.commands?.onCommand?.addListener((command) => {
    if (command === "toggle_pin") {
      toggleSidePanel().catch(error => {
        console.error('Keyboard shortcut error:', error);
      });
    }
  });

  // Track side panel state
  chrome.sidePanel?.onStatusChanged?.addListener(async (status) => {
    try {
      isPinned = status.isShown;
      await updatePinIcon();
    } catch (error) {
      console.error('Status change error:', error);
    }
  });
}

 // Separate async function for fetching shared secret
 async function fetchSharedSecret() {
  try {
    console.log('Fetching shared secret from API...');
    
    // Option 1: Send extensionId as a query parameter
    const url = new URL(`${API_CONFIG.BASE_PLAYGROUND_PATH}api/get-secret`);
    url.searchParams.append('extensionId', chrome.runtime.id);
    
    const response = await fetch(url, {
      method: "GET",
      headers: {
        ...API_CONFIG.getHeaders(),
      }
    });

    console.log('Response status:', response.status);

    if (!response.ok) {
      const errorText = await response.text();
      console.error('Error response:', errorText);
      throw new Error(`HTTP error! status: ${response.status}, details: ${errorText}`);
    }

    const data = await response.json();
    console.log('Response received successfully');

    if (!data || !data.shared_secret) {
      throw new Error('Invalid response format');
    }

    return data.shared_secret;
  } catch (error) {
    console.error('Failed to fetch shared secret:', error);
    throw error;
  }
}

chrome.runtime.onMessage.addListener(async (message, sender, sendResponse) => {
  console.log('Received message in background:', message);

  if (!message || typeof message.action !== 'string') {
    sendResponse({ success: false, error: 'Invalid message format' });
    return;
  }

  switch (message.action) {
    case 'getSharedSecret':
      try {
        console.log("Fetching shared secret...");
        const secret = await fetchSharedSecret();
        sendResponse({ secret });
      } catch (error) {
        console.error('Error fetching shared secret:', error);
        sendResponse({ error: error.message });
      }
      break;

    case 'getPinState':
      sendResponse({ isPinned });
      break;

    case 'updatePinState':
      isPinned = message.isPinned;
      updatePinIcon().catch(console.error);
      sendResponse({ success: true });
      break;

    case 'createDetachedWindow':
      handleDetachedWindow(message, sendResponse);
      break;

    case 'togglePin':
      handleTogglePin(sendResponse);
      break;

    case 'getLastActiveTab':
      handleGetLastActiveTab(sendResponse);
      break;

    default:
      sendResponse({ success: false, error: 'Unknown action' });
      return false; // Close the message channel
  }

  return true; // Keep the message channel open for async response
});