// content-script-loader.js
class ContentExtractor {
  static CONFIG = {
    TITLE_SELECTORS: [
      '.article-title',
      'h1.title',
      'h1.article-title',
      'h1.entry-title',
      '.post-title',
      'header h1',
      'article h1',
      'h1:first-of-type',
      'meta[property="og:title"]',
      'meta[name="twitter:title"]'
    ],
    MIN_TEXT_LENGTH: 10,
    MAX_TEXT_LENGTH: 50000,
    PRIORITY_SELECTORS: [
      '#article-text',
      'main',
      'article',
      '[role="main"]',
      '.content',
      '#content',
      'h1', 'h2', 'h3', 'h4', 'h5', 'h6',
      'p',
      'div.article-body',
      'div.entry-content'
    ],
    SKIP_SELECTORS: [
      // Base selectors
      'nav', 'header', 'footer', 'aside', 'sidebar',
      '[role="navigation"]',
      '[role="banner"]',
      '[role="contentinfo"]',
      '.navigation', '.menu', '.sidebar', '.footer', '.header',

      // Right section
      '#rightsection',
      '#rightsection *',
      '.right-artl',
      '[class*="right-artl"]',
      '.med_rec_lg_min',
      '[id*="gpt_unit_"]',
      '[id*="google_ads"]',
      'iframe',

      // Form and modal related
      '.form-container-Inn',
      '.forgot-modal',
      '.form-container-Inn.forgot-modal',
      '[id*="Form"]',
      '[id="forgotForm"]',
      '[class*="modal"]',
      '.Form-LeftSection',
      '.Form-RightSection',
      'form',
      '.loginForm',
      '.Text-container',
      '.Rinput1',
      '[id*="recaptcha"]',
      '[class*="login"]',
      '[class*="signin"]',
      '[class*="signup"]',
      '[class*="forgot"]',
      '[class*="reset"]',
      '#forgotForm',
      '.alert-area',

      // Advertisement related
      '.ad',
      '.advertisement',
      '[data-adpath]',
      '[data-google-query-id]',

      // Other content to skip
      '.social-share',
      '.comments',
      '.related-content',
      '.meta',
      '.byline',
      '.related-more-stories',
      '.related-more-stories *',
      'div[class*="related"]',
      '[class*="related-stories"]',
      '.ls-separate',
      '.option-bar',
      '[class*="option-bar"]',
      'div[class*="option-bar"]',
      'div[class*="option-bar"] *',
      '.article-left',
      '.article-right',
      '.floatLeft',
      '.floatRight',
      '.wp-block-list',
      '.related-more-stories'
    ],
    SKIP_KEYWORDS: [
      'login',
      'signin',
      'signup',
      'register',
      'forgot password',
      'reset password',
      'welcome',
      'sign in to continue',
      'already have an account',
      'email',
      'password',
      'submit',
      'recaptcha',
      'reset code',
      'send reset code',
      'goto-signin'
    ],

    // Word limit settings
    WORD_LIMITS: {
      MIN_WORDS: 5,          // Minimum words for a valid chunk
      MAX_WORDS: 1000,       // Maximum words for TTS processing
      OPTIMAL_WORDS: 800,    // Target word count for best TTS performance
      PARAGRAPH_MIN: 10      // Minimum words for a valid paragraph
    },

    // Content optimization
    CONTENT_PRIORITY: {
      TITLE_WEIGHT: 1.5,     // Weight for title relevance
      FIRST_PARA_WEIGHT: 1.2 // Weight for first paragraph
    },
    SENTENCE_BREAKS: ['. ', '? ', '! ', ';', '\n']
  };

  static shouldSkipElement(element) {
    let current = element;
    while (current && current !== document.body) {
      // 1. Check direct matches with skip selectors
      if (this.CONFIG.SKIP_SELECTORS.some(selector => {
        try {
          return current.matches(selector);
        } catch (e) {
          return false;
        }
      })) {
        return true;
      }

      // 2. Check class names
      if (current.className && typeof current.className === 'string') {
        const classNames = current.className.split(' ');
        if (classNames.some(className =>
          className.includes('right-artl') ||
          className.includes('med_rec') ||
          className.includes('gpt_unit') ||
          className.includes('google_ads') ||
          className.includes('form') ||
          className.includes('modal') ||
          className.includes('login') ||
          className.includes('signin') ||
          className.includes('forgot') ||
          className.includes('reset')
        )) {
          return true;
        }
      }

      // 3. Check element visibility
      const style = window.getComputedStyle(current);
      if (style.display === 'none' || style.visibility === 'hidden') {
        return true;
      }

      current = current.parentElement;
    }
    return false;
  }

  static extractByPrioritySelectors() {
    const contentElements = [];
    const processedElements = new Set();

    // First try to get content from #article-text
    const articleText = document.getElementById('article-text');
    if (articleText && !this.shouldSkipElement(articleText)) {
      const text = articleText.innerText.trim();
      if (this.isValidText(text)) {
        contentElements.push(text);
        processedElements.add(articleText);
        return contentElements;
      }
    }

    // Fallback to other selectors
    for (const selector of this.CONFIG.PRIORITY_SELECTORS) {
      const elements = document.querySelectorAll(selector);
      elements.forEach(element => {
        if (processedElements.has(element) || this.shouldSkipElement(element)) {
          return;
        }

        const text = element.innerText.trim();
        if (this.isValidText(text)) {
          contentElements.push(text);
          processedElements.add(element);
        }
      });
    }

    return contentElements;
  }

  static extractByTextDensity() {
    const allElements = document.getElementsByTagName('*');
    const contentCandidates = [];

    for (const element of allElements) {
      if (this.shouldSkipElement(element)) continue;

      const text = element.innerText.trim();
      if (!this.isValidText(text)) continue;

      const rect = element.getBoundingClientRect();
      const area = rect.width * rect.height;
      const textDensity = text.length / (area || 1);

      contentCandidates.push({ text, density: textDensity, element });
    }

    return contentCandidates
      .sort((a, b) => b.density - a.density)
      .slice(0, 5)
      .map(candidate => candidate.text);
  }

  static extractTitle() {
    // First try direct title selectors
    for (const selector of this.CONFIG.TITLE_SELECTORS) {
      const titleElement = document.querySelector(selector);
      if (titleElement) {
        // For meta tags, get the content attribute
        if (titleElement.tagName.toLowerCase() === 'meta') {
          const content = titleElement.getAttribute('content');
          if (content) return content.trim();
        } else {
          const text = titleElement.innerText;
          if (text) return text.trim();
        }
      }
    }
    return null;
  }

  static optimizeExtractedContent(extractedContent) {
    let wordCount = 0;
    let optimizedContent = [];

    // Preprocess text similar to TextStreamManager
    const preprocessText = (text) => {
        return text
            .replace(/\.(?=[A-Z])/g, '. ')  // Add space after period before capital letters
            .replace(/\s+/g, ' ')           // Normalize whitespace
            .replace(/\n\s*\n/g, '\n')      // Normalize multiple newlines
            .trim();
    };

    for (const section of extractedContent) {
        // Preprocess the section text
        const processedSection = preprocessText(section);
        
        // Split into paragraphs
        const paragraphs = processedSection.split(/\n\s*\n/);

        for (const paragraph of paragraphs) {
            const trimmedPara = paragraph.trim();
            if (!trimmedPara) continue;

            // Count words in this paragraph
            const paraWords = trimmedPara.split(/\s+/).length;

            // Skip very short paragraphs
            if (paraWords < this.CONFIG.WORD_LIMITS.MIN_PARAGRAPH) {
                continue;
            }

            // Check if adding would exceed limit
            if (wordCount + paraWords > this.CONFIG.WORD_LIMITS.MAX_WORDS) {
                // Only try to add partial content if we haven't reached optimal length
                if (wordCount < this.CONFIG.WORD_LIMITS.OPTIMAL_WORDS) {
                    const remainingWords = this.CONFIG.WORD_LIMITS.MAX_WORDS - wordCount;
                    
                    // Find a good breaking point
                    let partialText = '';
                    const words = trimmedPara.split(/\s+/);
                    let currentWordCount = 0;
                    
                    for (const word of words) {
                        if (currentWordCount + 1 > remainingWords) break;
                        
                        const potentialText = partialText + (partialText ? ' ' : '') + word;
                        const lastBreak = this.findLastSentenceBreak(potentialText);
                        
                        if (lastBreak) {
                            partialText = potentialText;
                            currentWordCount++;
                        } else {
                            // If no sentence break found and we're at a good length,
                            // add a period and break
                            if (currentWordCount > this.CONFIG.WORD_LIMITS.MIN_WORDS) {
                                partialText = potentialText + '.';
                                break;
                            }
                            partialText = potentialText;
                            currentWordCount++;
                        }
                    }
                    
                    if (partialText) {
                        optimizedContent.push(partialText.trim());
                        wordCount += currentWordCount;
                    }
                }
                break;
            }

            // Process the full paragraph
            let paraText = trimmedPara;
            const lastBreak = this.findLastSentenceBreak(paraText);
            
            // Ensure the paragraph ends with proper punctuation
            if (!lastBreak || !lastBreak.break.trim()) {
                paraText += '.';
            }

            optimizedContent.push(paraText);
            wordCount += paraWords;

            if (wordCount >= this.CONFIG.WORD_LIMITS.MAX_WORDS) {
                break;
            }
        }

        if (wordCount >= this.CONFIG.WORD_LIMITS.MAX_WORDS) {
            break;
        }
    }

    return optimizedContent;
}

  // Helper method to ensure text ends with proper punctuation
  static ensureProperEnding(text) {
    const trimmedText = text.trim();

    // Check if the text already ends with any sentence break
    const hasProperEnding = this.CONFIG.SENTENCE_BREAKS.some(breakChar =>
      trimmedText.endsWith(breakChar.trim())
    );

    return hasProperEnding ? trimmedText : trimmedText + '.';
  }

  // Helper method to find the last sentence break (from existing code)
  static findLastSentenceBreak(text) {
    let lastIndex = -1;
    let lastBreak = null;

    for (const breakChar of this.CONFIG.SENTENCE_BREAKS) {
      const index = text.lastIndexOf(breakChar);
      if (index > lastIndex) {
        lastIndex = index;
        lastBreak = breakChar;
      }
    }

    return lastIndex > -1 ? { index: lastIndex + lastBreak.length, break: lastBreak } : null;
  }

  static extractPageContent() {
    const extractionStrategies = [
      this.extractByPrioritySelectors.bind(this),
      this.extractByTextDensity.bind(this)
    ];

    for (const strategy of extractionStrategies) {
      const extractedContent = strategy();
      if (extractedContent.length > 0) {
        // Optimize the extracted content
        const optimizedContent = this.optimizeExtractedContent(extractedContent);
        return optimizedContent;
      }
    }

    // Fallback to body text with optimization
    return this.optimizeExtractedContent([document.body.innerText]);
  }



  static isValidText(text) {
    return text &&
      text.length >= this.CONFIG.MIN_TEXT_LENGTH &&
      text.length <= this.CONFIG.MAX_TEXT_LENGTH &&
      !this.hasNavigationKeywords(text);
  }

  static hasNavigationKeywords(text) {
    const navigationWords = [
      'menu',
      'navigation',
      'footer',
      'sidebar',
      'copyright',
      'login',
      'signup',
      'advertisement',
      'related content',
      'more articles',
      'continue reading'
    ];

    const lowerText = text.toLowerCase();
    return navigationWords.some(word => lowerText.includes(word));
  }
}

// Message listener for extension communication
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  console.log('Content script received message:', request);

  switch (request.action) {
    case "getPageText":
      try {
        const content = ContentExtractor.extractPageContent();
        const title = ContentExtractor.extractTitle();
        const pageText = [title, ...content].filter(Boolean).join('\n\n');
        console.log('Extracted page text length:', pageText.length);
        sendResponse({ success: true, pageText });
      } catch (error) {
        console.error('Error extracting page text:', error);
        sendResponse({ success: false, error: error.message });
      }
      break;

    default:
      console.log('Unknown action received:', request.action);
      sendResponse({ success: false, error: 'Unknown action' });
  }
  return true;
});