// // content.js

import { TextStreamManager, PageContentExtractor } from './TextStreamManager.js';

console.log("Content script starting...");

// Global variables
let selectedText = '';
let handleTTS, pauseTTS, resumeTTS, stopTTS;
let currentAudio = null;
let textStreamManager = null;
let isStreaming = false;

// Message handler
async function handleMessage(message, sender, sendResponse) {
    console.log("Received message in content script:", message);

    if (!message || !message.action) {
        console.error('Invalid message received');
        sendResponse({ success: false, error: 'Invalid message' });
        return true;
    }

    try {
        switch (message.action) {
            case "startReading":
            case "abenaReadAloud":
                await handleStartReading(message.text, sendResponse);
                break;

            case "stopReading":
            case "executeStopTTS":
                await handleStopReading();
                sendResponse({ success: true });
                break;

            case "pauseReading":
            case "executePauseTTS":
                await handlePauseReading();
                sendResponse({ success: true });
                break;

            case "resumeReading":
            case "executeResumeTTS":
                await handleResumeReading();
                sendResponse({ success: true });
                break;

            case "getPageText":
                const pageText = PageContentExtractor.getPrioritizedContent().join('\n\n');
                sendResponse({
                    success: true,
                    pageText: pageText
                });
                break;

            default:
                console.warn('Unknown message action:', message.action);
                sendResponse({ success: false, error: 'Unknown action' });
        }
    } catch (error) {
        console.error('Error handling message:', error);
        sendResponse({ success: false, error: error.message });
    }

    return true;
}

// Set up both message listeners for compatibility
chrome.runtime.onMessage.addListener(handleMessage);
window.addEventListener('message', (event) => {
    if (event.data.type === 'FROM_EXTENSION') {
        handleMessage(event.data.message, null, (response) => {
            window.postMessage({
                type: 'TO_EXTENSION',
                response: response
            }, '*');
        });
    }
});

async function handleStartReading(text, sendResponse) {
    if (!handleTTS || !textStreamManager) {
        throw new Error('TTS system not initialized');
    }

    // Stop any current playback
    await handleStopReading();

    // Start new reading
    isStreaming = true;
    console.log('Starting text stream processing');

    try {
        await textStreamManager.processTextStream(text);
        console.log('Text stream processing completed');
        notifyExtensionOfCompletion();
        if (sendResponse) sendResponse({ success: true });
    } catch (error) {
        console.error('Error processing text stream:', error);
        notifyExtensionOfError(error.message);
        if (sendResponse) sendResponse({ success: false, error: error.message });
    } finally {
        isStreaming = false;
    }
}

async function handleStopReading() {
    isStreaming = false;
    if (textStreamManager) {
        textStreamManager.stop();
    }
    if (currentAudio) {
        currentAudio.pause();
        currentAudio = null;
    }
}

async function handlePauseReading() {
    if (textStreamManager) {
        textStreamManager.pause();
    }
    if (currentAudio) {
        currentAudio.pause();
    }
}

async function handleResumeReading() {
    if (textStreamManager) {
        textStreamManager.resume();
    }
    if (currentAudio) {
        await currentAudio.play().catch(console.error);
    }
}

function notifyExtensionOfCompletion() {
    chrome.runtime.sendMessage({
        action: "speechEnded",
        success: true
    }).catch(console.error);
}

function notifyExtensionOfError(error) {
    chrome.runtime.sendMessage({
        action: "speechError",
        error: error
    }).catch(console.error);
}

// Initialize TextStreamManager
function initializeStreamManager() {
    if (textStreamManager) return;

    console.log('Initializing TextStreamManager');
    textStreamManager = new TextStreamManager(
        async (chunk) => {
            if (!isStreaming) {
                console.log('Streaming disabled');
                return false;
            }

            try {
                console.log('Processing chunk:', chunk.substring(0, 50) + '...');
                const audio = await handleTTS(chunk);
                console.log('Chunk processing complete');
                return true;
            } catch (error) {
                console.error('Error processing chunk:', error);
                return false;
            }
        },
        {
            minChunkSize: 50,
            // maxChunkSize: 150,
            maxChunkSize: 250,
            chunkDelay: 250,
            delimiters: ['. ', '? ', '! ', ';', '\n'],
            debug: true
        }
    );
}

// Initialize when the script loads
(async () => {
    try {
        // Import required modules
        const module = await import(chrome.runtime.getURL('shared.js'));
        
        // Store the TTS functions in global scope
        handleTTS = module.handleTTS;
        pauseTTS = module.pauseTTS;
        resumeTTS = module.resumeTTS;
        stopTTS = module.stopTTS;

        console.log('TTS functions imported successfully');

        // Initialize TextStreamManager
        initializeStreamManager();

        console.log('Content script initialized successfully');
    } catch (error) {
        console.error('Error initializing content script:', error);
    }
})();